%
% function x = horibas(value,range,species)
%
% Inputs: value   = reading from analyzer (value <= 1)
%         range   = range at which the reading was taken
%         species = one of the following:
%                   - 'THC'  = total hydrocarbons
%                   - 'loCO' = CO in the low range
%                   - 'hiCO' = CO in the high range
%                   - 'O2'   = O2
%                   - 'CO2'  = CO2
%                   - 'NOx'  = NO + NO2 + NO3
%
%         NOTE: value and range can be arrays, but must
%               be of same size
%
% Output: x = mole fraction, [-]
%
% Credits
% Prepared by Fabrizio Bisetti (fbisetti@me.berkeley.edu) using
% Horiba Instruments Inc. calibration tables and graphs produced
% by Scott G. McCormick (Dept. Mechanical Eng., UC Berkeley).
%
%

% ME107B -- Canister Combustor -- Spring 2006
% Calibration of Horibas

function x = horibas(value,range,species)

% species available
chemical = {'THC','loCO','hiCO','O2','CO2','NOx'};

% identify species required by user
i=1;
while i <= length(chemical)
  if strcmp(species,chemical{i})
      break
  end
  i=i+1;
end

% check for errors
if i==length(chemical)+1
    error('Species not available');
end

% check sizes
if length(value)~=length(range)
  error('If arrays, value and range must have the same length');
end

% multiply by 100 to get 100%
% NOTE: the formulas given with the horibas
%       work with % values as input. This matlab
%       function, instead, takes the reading
%       on the display, which is a fraction of 1.0

read=value*100;


% repeat for every reading
for k=1:length(read)
    
    %-----------------------------------------------------------------------    
    % i=1: THC
    %-----------------------------------------------------------------------
    % NOTE: the formulas give THC in ppm. Here we convert directly
    %       to mole fraction

    if i==1
        switch range(k)
            case 1
                % range 1
                A=0;
                B=0.1;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 2
                % range 2
                A=0;
                B=0.3;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 3
                % range 3
                A=0;
                B=1.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 4
                % range 4
                A=0;
                B=3.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 5
                % range 5
                A=0;
                B=10.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;                
            case 6
                % range 6
                A=0;
                B=30.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;                
            case 7
                % range 7
                A=0;
                B=100.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;                
            case 8
                % range 8
                A=0;
                B=300.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            otherwise
                error('Unknown range');        
        end
    end

    
    %-----------------------------------------------------------------------
    % i=2: loCO
    %-----------------------------------------------------------------------
    % NOTE: the formulas give low CO in ppm. Here we convert directly
    %       to mole fraction
    
    if i==2
        switch range(k)
            case 1
                % range 1
                A=-3.164977E-3;
                B=1.006381;
                C=-4.642649E-4;
                D=9.02655E-6;
                E=-5.01218E-8;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 2
                % range 2
                A=-7.295325E-3;
                B=2.864412;
                C=-1.499218E-3;
                D=5.22444E-5;
                E=-2.368713E-7;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 3
                % range 3
                A=1.353769E-2;
                B=7.687434;
                C=1.247239E-2;
                D=1.464194E-4;
                E=-3.97985E-7;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 4
                % range 4
                A=4.161048E-2;
                B=14.8356;
                C=9.882978E-2;
                D=-6.721495E-5;
                E=5.922779E-6;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            otherwise
                error('Unknown range');        
        end
    end
    
    
    %-----------------------------------------------------------------------
    % i=3: hiCO
    %-----------------------------------------------------------------------
    % NOTE: the formulas give high CO in %. Here we convert directly
    %       to mole fraction
    
    if i==3
        switch range(k)
            case 1
                % range 1
                A=2.418589E-4;
                B=4.709818E-3;
                C=2.877999E-6;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 2
                % range 2
                A=4.332392E-4;
                B=1.613321E-2;
                C=2.974977E-5;
                D=8.874444E-8;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 3
                % range 3
                A=-3.381558E-4;
                B=2.971657E-2;
                C=1.138822E-4;
                D=8.899061E-7;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 4
                % range 4
                A=-6.338745E-3;
                B=4.669954E-2;
                C=8.360011E-5;
                D=4.500329E-6;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            otherwise
                error('Unknown range');        
        end
    end
    
    %-----------------------------------------------------------------------    
    % i=4: O2
    %-----------------------------------------------------------------------
    % NOTE: the formulas give high O2 in %. Here we convert directly
    %       to mole fraction
    if i==4
        switch range(k)
            case 1
                % range 1
                A=0;
                B=0.05;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 2
                % range 2
                A=0;
                B=0.1;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 3
                % range 3
                A=0;
                B=0.25;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            otherwise
                error('Unknown range');
        end
    end

    %-------------------------------------------------------------------
    % i=5: CO2 w/ span gas 3.8%
    %-------------------------------------------------------------------
    % NOTE: the formulas give high CO2 in %. Here we convert directly
    %       to mole fraction
    
    if i==5
        switch range(k)
            case 1
                % range 1
                A=-2.256585E-4;
                B=9.579003E-3;
                C=4.232841E-6;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 2
                % range 2
                A=2.240643E-3;
                B=3.138901E-2;
                C=8.588708E-5;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 3
                % range 3
                A=-3.074441E-3;
                B=5.414548E-2;
                C=1.344948E-4;
                D=1.235553E-6;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;
            case 4
                % range 4
                A=-2.543291E-2;
                B=9.025789E-2;
                C=-5.051354E-6;
                D=7.442171E-6;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^2;        
            otherwise
                error('Unknown range');        
        end
    end

   
    %-------------------------------------------------------------------
    % i=6: NOx
    %-------------------------------------------------------------------

    if i==6
        switch range(k)
            case 2
                % range 2
                A=0;
                B=0.3;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 3
                % range 3
                A=0;
                B=1.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 4
                % range 4
                A=0;
                B=3.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;
            case 5
                % range 5
                A=0;
                B=10.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;                
            case 6
                % range 6
                A=0;
                B=30.0;
                C=0;
                D=0;
                E=0;
                x(k)=(A+B*read(k)+C*read(k)^2+...
                    D*read(k)^3+E*read(k)^4)/10^6;                
            otherwise
                error('Unknown range');        
        end
    end
end